#!/usr/bin/env python3
"""
LVGL Image Converter - Python Version
Converts images to various color formats supported by LVGL with optional dithering.
"""

import os
import sys
import argparse
import struct
from PIL import Image
import numpy as np


class LVGLConverter:
    """Image converter for LVGL graphics library"""
    
    # Color format constants
    CF_TRUE_COLOR_332 = 0
    CF_TRUE_COLOR_565 = 1
    CF_TRUE_COLOR_565_SWAP = 2
    CF_TRUE_COLOR_888 = 3
    CF_ALPHA_1_BIT = 4
    CF_ALPHA_2_BIT = 5
    CF_ALPHA_4_BIT = 6
    CF_ALPHA_8_BIT = 7
    CF_INDEXED_1_BIT = 8
    CF_INDEXED_2_BIT = 9
    CF_INDEXED_4_BIT = 10
    CF_INDEXED_8_BIT = 11
    CF_RAW = 12
    CF_RAW_ALPHA = 13
    CF_RAW_CHROMA = 14
    
    # Helper formats for output
    CF_TRUE_COLOR = 100
    CF_TRUE_COLOR_ALPHA = 101
    CF_TRUE_COLOR_CHROMA = 102
    
    def __init__(self, image_path, output_name, dithering=False, color_format="true_color"):
        """
        Initialize the converter
        
        Args:
            image_path (str): Path to the input image
            output_name (str): Name for the output file (without extension)
            dithering (bool): Enable/disable dithering
            color_format (str): Color format for conversion
        """
        self.path = image_path
        self.out_name = output_name
        self.dith = dithering
        self.cf_string = color_format
        self.alpha = False
        self.chroma = False
        self.d_out = []
        
        # Load image
        try:
            self.img = Image.open(image_path).convert('RGBA')
            self.w, self.h = self.img.size
        except Exception as e:
            raise ValueError(f"Failed to load image: {e}")
        
        # Initialize dithering arrays
        self.r_earr = [0] * (self.w + 2) if self.dith else None
        self.g_earr = [0] * (self.w + 2) if self.dith else None
        self.b_earr = [0] * (self.w + 2) if self.dith else None
        
        self.r_nerr = 0
        self.g_nerr = 0
        self.b_nerr = 0
        
    def convert(self, cf, alpha=False):
        """Convert image to specified color format"""
        self.cf = cf
        self.alpha = alpha
        self.d_out = []
        
        # Handle raw formats
        if cf in [self.CF_RAW, self.CF_RAW_ALPHA, self.CF_RAW_CHROMA]:
            with open(self.path, 'rb') as f:
                data = f.read()
                self.d_out = list(data)
            return
        
        # Handle indexed formats - create palette first
        palette_size = 0
        if cf == self.CF_INDEXED_1_BIT:
            palette_size = 2
        elif cf == self.CF_INDEXED_2_BIT:
            palette_size = 4
        elif cf == self.CF_INDEXED_4_BIT:
            palette_size = 16
        elif cf == self.CF_INDEXED_8_BIT:
            palette_size = 256
        
        if palette_size > 0:
            # Convert to indexed color
            img_rgb = self.img.convert('RGB')
            img_quantized = img_rgb.quantize(colors=palette_size)
            palette = img_quantized.getpalette()
            
            # Add palette to output
            for i in range(palette_size):
                if palette and i * 3 + 2 < len(palette):
                    r, g, b = palette[i*3:i*3+3]
                    self.d_out.extend([b, g, r, 0xFF])  # BGRA format
                else:
                    self.d_out.extend([0xFF, 0xFF, 0xFF, 0xFF])
            
            # Store the quantized image for pixel indexing
            self.quantized_img = img_quantized
        
        # Convert pixels
        pixels = np.array(self.img)
        for y in range(self.h):
            self._dith_reset()
            for x in range(self.w):
                self._conv_px(x, y, pixels[y, x])
    
    def _conv_px(self, x, y, pixel):
        """Convert a single pixel"""
        r, g, b, a = pixel
        
        # Handle alpha
        if self.alpha:
            a = 255 - ((255 - a) >> 1)  # Convert to 7-bit and invert
        else:
            a = 0xFF
        
        # Apply dithering
        self._dith_next(r, g, b, x)
        
        # Convert based on format
        if self.cf == self.CF_TRUE_COLOR_332:
            c8 = (self.r_act & 0xE0) | ((self.g_act & 0xE0) >> 3) | ((self.b_act & 0xC0) >> 6)
            self.d_out.append(c8)
            if self.alpha:
                self.d_out.append(a)
                
        elif self.cf == self.CF_TRUE_COLOR_565:
            c16 = ((self.r_act & 0xF8) << 8) | ((self.g_act & 0xFC) << 3) | ((self.b_act & 0xF8) >> 3)
            self.d_out.extend([c16 & 0xFF, (c16 >> 8) & 0xFF])
            if self.alpha:
                self.d_out.append(a)
                
        elif self.cf == self.CF_TRUE_COLOR_565_SWAP:
            c16 = ((self.r_act & 0xF8) << 8) | ((self.g_act & 0xFC) << 3) | ((self.b_act & 0xF8) >> 3)
            self.d_out.extend([(c16 >> 8) & 0xFF, c16 & 0xFF])
            if self.alpha:
                self.d_out.append(a)
                
        elif self.cf == self.CF_TRUE_COLOR_888:
            self.d_out.extend([self.b_act, self.g_act, self.r_act, a])
            
        elif self.cf == self.CF_ALPHA_1_BIT:
            w = (self.w + 7) // 8
            p = w * y + (x // 8)
            while len(self.d_out) <= p:
                self.d_out.append(0)
            if a > 0x80:
                self.d_out[p] |= 1 << (7 - (x & 0x7))
                
        elif self.cf == self.CF_ALPHA_2_BIT:
            w = (self.w + 3) // 4
            p = w * y + (x // 4)
            while len(self.d_out) <= p:
                self.d_out.append(0)
            self.d_out[p] |= (a >> 6) << (6 - ((x & 0x3) * 2))
            
        elif self.cf == self.CF_ALPHA_4_BIT:
            w = (self.w + 1) // 2
            p = w * y + (x // 2)
            while len(self.d_out) <= p:
                self.d_out.append(0)
            self.d_out[p] |= (a >> 4) << (4 - ((x & 0x1) * 4))
            
        elif self.cf == self.CF_ALPHA_8_BIT:
            p = self.w * y + x
            while len(self.d_out) <= p:
                self.d_out.append(0)
            self.d_out[p] = a
            
        # Handle indexed formats
        elif self.cf in [self.CF_INDEXED_1_BIT, self.CF_INDEXED_2_BIT, 
                        self.CF_INDEXED_4_BIT, self.CF_INDEXED_8_BIT]:
            # Get color index from quantized image
            color_index = self._get_color_index(x, y)
            self._store_indexed_pixel(x, y, color_index)
    
    def _get_color_index(self, x, y):
        """Get color index for indexed formats"""
        #if hasattr(self.img, 'mode') and self.img.mode == 'P':
        #    return self.img.getpixel((x, y))
        if hasattr(self.img, 'mode'):
            return self.quantized_img.getpixel((x, y))
        return 0
    
    def _store_indexed_pixel(self, x, y, color_index):
        """Store pixel for indexed formats"""
        if self.cf == self.CF_INDEXED_1_BIT:
            w = (self.w + 7) // 8
            p = w * y + (x // 8) + 8  # +8 for palette
            while len(self.d_out) <= p:
                self.d_out.append(0)
            self.d_out[p] |= (color_index & 0x1) << (7 - (x & 0x7))
            
        elif self.cf == self.CF_INDEXED_2_BIT:
            w = (self.w + 3) // 4
            p = w * y + (x // 4) + 16  # +16 for palette
            while len(self.d_out) <= p:
                self.d_out.append(0)
            self.d_out[p] |= (color_index & 0x3) << (6 - ((x & 0x3) * 2))
            
        elif self.cf == self.CF_INDEXED_4_BIT:
            w = (self.w + 1) // 2
            p = w * y + (x // 2) + 64  # +64 for palette
            while len(self.d_out) <= p:
                self.d_out.append(0)
            self.d_out[p] |= (color_index & 0xF) << (4 - ((x & 0x1) * 4))
            
        elif self.cf == self.CF_INDEXED_8_BIT:
            p = self.w * y + x + 1024  # +1024 for palette
            while len(self.d_out) <= p:
                self.d_out.append(0)
            self.d_out[p] = color_index & 0xFF
    
    def _dith_reset(self):
        """Reset dithering for new row"""
        if self.dith:
            self.r_nerr = 0
            self.g_nerr = 0
            self.b_nerr = 0
    
    def _dith_next(self, r, g, b, x):
        """Apply dithering to pixel"""
        if self.dith:
            # Add errors from previous pixels
            self.r_act = min(255, max(0, r + self.r_nerr + self.r_earr[x+1]))
            self.g_act = min(255, max(0, g + self.g_nerr + self.g_earr[x+1]))
            self.b_act = min(255, max(0, b + self.b_nerr + self.b_earr[x+1]))
            
            self.r_earr[x+1] = 0
            self.g_earr[x+1] = 0
            self.b_earr[x+1] = 0
            
            # Quantize colors
            if self.cf == self.CF_TRUE_COLOR_332:
                self.r_act = self._classify_pixel(self.r_act, 3)
                self.g_act = self._classify_pixel(self.g_act, 3)
                self.b_act = self._classify_pixel(self.b_act, 2)
            elif self.cf in [self.CF_TRUE_COLOR_565, self.CF_TRUE_COLOR_565_SWAP]:
                self.r_act = self._classify_pixel(self.r_act, 5)
                self.g_act = self._classify_pixel(self.g_act, 6)
                self.b_act = self._classify_pixel(self.b_act, 5)
            elif self.cf == self.CF_TRUE_COLOR_888:
                self.r_act = self._classify_pixel(self.r_act, 8)
                self.g_act = self._classify_pixel(self.g_act, 8)
                self.b_act = self._classify_pixel(self.b_act, 8)
            
            # Calculate and distribute error
            r_err = r - self.r_act
            g_err = g - self.g_act
            b_err = b - self.b_act
            
            # Floyd-Steinberg dithering distribution
            self.r_nerr = round((7 * r_err) / 16)
            self.g_nerr = round((7 * g_err) / 16)
            self.b_nerr = round((7 * b_err) / 16)
            
            if x > 0:
                self.r_earr[x] += round((3 * r_err) / 16)
                self.g_earr[x] += round((3 * g_err) / 16)
                self.b_earr[x] += round((3 * b_err) / 16)
            
            self.r_earr[x+1] += round((5 * r_err) / 16)
            self.g_earr[x+1] += round((5 * g_err) / 16)
            self.b_earr[x+1] += round((5 * b_err) / 16)
            
            if x < self.w - 1:
                self.r_earr[x+2] += round(r_err / 16)
                self.g_earr[x+2] += round(g_err / 16)
                self.b_earr[x+2] += round(b_err / 16)
        else:
            # No dithering - just quantize
            if self.cf == self.CF_TRUE_COLOR_332:
                self.r_act = self._classify_pixel(r, 3)
                self.g_act = self._classify_pixel(g, 3)
                self.b_act = self._classify_pixel(b, 2)
            elif self.cf in [self.CF_TRUE_COLOR_565, self.CF_TRUE_COLOR_565_SWAP]:
                self.r_act = self._classify_pixel(r, 5)
                self.g_act = self._classify_pixel(g, 6)
                self.b_act = self._classify_pixel(b, 5)
            elif self.cf == self.CF_TRUE_COLOR_888:
                self.r_act = self._classify_pixel(r, 8)
                self.g_act = self._classify_pixel(g, 8)
                self.b_act = self._classify_pixel(b, 8)
    
    def _classify_pixel(self, value, bits):
        """Quantize pixel value to specified bit depth"""
        step = 1 << (8 - bits)
        return max(0, min(255, round(value / step) * step))
    
    def format_to_c_array(self):
        """Format output data as C array"""
        c_array = ""
        
        # Add format-specific headers
        if self.cf == self.CF_TRUE_COLOR_332:
            c_array += "\n#if LV_COLOR_DEPTH == 1 || LV_COLOR_DEPTH == 8"
            if not self.alpha:
                c_array += "\n  /*Pixel format: Blue: 2 bit, Green: 3 bit, Red: 3 bit*/"
            else:
                c_array += "\n  /*Pixel format: Blue: 2 bit, Green: 3 bit, Red: 3 bit, Alpha 8 bit */"
        elif self.cf == self.CF_TRUE_COLOR_565:
            c_array += "\n#if LV_COLOR_DEPTH == 16 && LV_COLOR_16_SWAP == 0"
            if not self.alpha:
                c_array += "\n  /*Pixel format: Blue: 5 bit, Green: 6 bit, Red: 5 bit*/"
            else:
                c_array += "\n  /*Pixel format: Blue: 5 bit, Green: 6 bit, Red: 5 bit, Alpha 8 bit*/"
        elif self.cf == self.CF_TRUE_COLOR_565_SWAP:
            c_array += "\n#if LV_COLOR_DEPTH == 16 && LV_COLOR_16_SWAP != 0"
            if not self.alpha:
                c_array += "\n  /*Pixel format: Blue: 5 bit, Green: 6 bit, Red: 5 bit BUT the 2 bytes are swapped*/"
            else:
                c_array += "\n  /*Pixel format: Blue: 5 bit Green: 6 bit, Red: 5 bit, Alpha 8 bit BUT the 2 color bytes are swapped*/"
        elif self.cf == self.CF_TRUE_COLOR_888:
            c_array += "\n#if LV_COLOR_DEPTH == 32"
            if not self.alpha:
                c_array += "\n  /*Pixel format: Blue: 8 bit, Green: 8 bit, Red: 8 bit, Fix 0xFF: 8 bit, */"
            else:
                c_array += "\n  /*Pixel format: Blue: 8 bit, Green: 8 bit, Red: 8 bit, Alpha: 8 bit*/"
        
        # Output palette for indexed formats
        if self.cf in [self.CF_INDEXED_1_BIT, self.CF_INDEXED_2_BIT, 
                      self.CF_INDEXED_4_BIT, self.CF_INDEXED_8_BIT]:
            palette_sizes = {
                self.CF_INDEXED_1_BIT: 2,
                self.CF_INDEXED_2_BIT: 4,
                self.CF_INDEXED_4_BIT: 16,
                self.CF_INDEXED_8_BIT: 256
            }
            palette_size = palette_sizes[self.cf]
            
            c_array += "\n"
            if self.cf_string != "indexed_2":
                for p in range(palette_size):
                    idx = p * 4
                    if idx + 3 < len(self.d_out):
                        c_array += f"  0x{self.d_out[idx]:02X}, 0x{self.d_out[idx+1]:02X}, "
                        c_array += f"0x{self.d_out[idx+2]:02X}, 0x{self.d_out[idx+3]:02X}, "
                        c_array += f"\t/*Color of index {p}*/\n"
            else:
                c_array += f"  0xff, 0xff, 0xff, 0xff,  /* index 0 = white */\n"
                c_array += f"  0x00, 0x00, 0xff, 0xff,  /* index 1 = red */\n"
                c_array += f"  0x00, 0x00, 0x00, 0xff,  /* index 2 = black */\n"
                c_array += f"  0x00, 0x00, 0x00, 0x00,  /* index 3 = unused (transparent) */\n"
        
        # Output pixel data
        start_idx = 0
        if self.cf == self.CF_INDEXED_1_BIT:
            start_idx = 8
        elif self.cf == self.CF_INDEXED_2_BIT:
            start_idx = 16
        elif self.cf == self.CF_INDEXED_4_BIT:
            start_idx = 64
        elif self.cf == self.CF_INDEXED_8_BIT:
            start_idx = 1024
        
        # Format data output
        c_array += "\n  "
        for i, byte_val in enumerate(self.d_out[start_idx:], start_idx):
            if i > start_idx and (i - start_idx) % (100) == 0:
                c_array += "\n  "
            c_array += f"0x{byte_val:02X}, "
        
        if self.cf in [self.CF_TRUE_COLOR_332, self.CF_TRUE_COLOR_565, 
                      self.CF_TRUE_COLOR_565_SWAP, self.CF_TRUE_COLOR_888]:
            c_array += "\n#endif"
        
        return c_array
    
    def get_c_header(self):
        """Generate C header"""
        attr_name = f"LV_ATTRIBUTE_IMG_{self.out_name.upper()}"
        
        header = f'''#ifdef __has_include
    #if __has_include("lvgl.h")
        #ifndef LV_LVGL_H_INCLUDE_SIMPLE
            #define LV_LVGL_H_INCLUDE_SIMPLE
        #endif
    #endif
#endif

#if defined(LV_LVGL_H_INCLUDE_SIMPLE)
    #include "lvgl.h"
#else
    #include "lvgl/lvgl.h"
#endif

#ifndef {attr_name}
#define {attr_name}
#endif

const LV_ATTRIBUTE_MEM_ALIGN LV_ATTRIBUTE_LARGE_CONST {attr_name} uint8_t {self.out_name}_map[] = {{'''
        
        return header
    
    def get_c_footer(self, cf):
        """Generate C footer"""
        cf_map = {
            self.CF_TRUE_COLOR: ("LV_IMG_CF_TRUE_COLOR", f"{self.w * self.h} * LV_COLOR_SIZE / 8"),
            self.CF_TRUE_COLOR_ALPHA: ("LV_IMG_CF_TRUE_COLOR_ALPHA", f"{self.w * self.h} * LV_IMG_PX_SIZE_ALPHA_BYTE"),
            self.CF_TRUE_COLOR_CHROMA: ("LV_IMG_CF_TRUE_COLOR_CHROMA_KEYED", f"{self.w * self.h} * LV_COLOR_SIZE / 8"),
            self.CF_ALPHA_1_BIT: ("LV_IMG_CF_ALPHA_1BIT", str(len(self.d_out))),
            self.CF_ALPHA_2_BIT: ("LV_IMG_CF_ALPHA_2BIT", str(len(self.d_out))),
            self.CF_ALPHA_4_BIT: ("LV_IMG_CF_ALPHA_4BIT", str(len(self.d_out))),
            self.CF_ALPHA_8_BIT: ("LV_IMG_CF_ALPHA_8BIT", str(len(self.d_out))),
            self.CF_INDEXED_1_BIT: ("LV_IMG_CF_INDEXED_1BIT", str(len(self.d_out))),
            self.CF_INDEXED_2_BIT: ("LV_IMG_CF_INDEXED_2BIT", str(len(self.d_out))),
            self.CF_INDEXED_4_BIT: ("LV_IMG_CF_INDEXED_4BIT", str(len(self.d_out))),
            self.CF_INDEXED_8_BIT: ("LV_IMG_CF_INDEXED_8BIT", str(len(self.d_out))),
            self.CF_RAW: ("LV_IMG_CF_RAW", str(len(self.d_out))),
            self.CF_RAW_ALPHA: ("LV_IMG_CF_RAW_ALPHA", str(len(self.d_out))),
            self.CF_RAW_CHROMA: ("LV_IMG_CF_RAW_CHROMA_KEYED", str(len(self.d_out))),
        }
        
        cf_name, data_size = cf_map.get(cf, ("LV_IMG_CF_TRUE_COLOR", str(len(self.d_out))))
        
        footer = f'''
}};

const lv_img_dsc_t {self.out_name} = {{
  .header.always_zero = 0,
  .header.w = {self.w},
  .header.h = {self.h},
  .data_size = {data_size},
  .header.cf = {cf_name},
  .data = {self.out_name}_map,
}};
'''
        return footer
    
    def save_c_array(self, filename=None):
        """Save as C array file"""
        if filename is None:
            filename = f"{self.out_name}.c"
        
        alpha = "alpha" in self.cf_string
        
        if self.cf_string.startswith("true_color"):
            # Generate all color depths for true color
            formats = [
                (self.CF_TRUE_COLOR_332, alpha),
                (self.CF_TRUE_COLOR_565, alpha),
                (self.CF_TRUE_COLOR_565_SWAP, alpha),
                (self.CF_TRUE_COLOR_888, alpha)
            ]
            
            c_content = ""
            for fmt, use_alpha in formats:
                self.convert(fmt, use_alpha)
                c_content += self.format_to_c_array()
            
            # Determine output format
            if "alpha" in self.cf_string:
                output_cf = self.CF_TRUE_COLOR_ALPHA
            elif "chroma" in self.cf_string:
                output_cf = self.CF_TRUE_COLOR_CHROMA
            else:
                output_cf = self.CF_TRUE_COLOR
        else:
            # Single format conversion
            format_map = {
                "alpha_1": self.CF_ALPHA_1_BIT,
                "alpha_2": self.CF_ALPHA_2_BIT,
                "alpha_4": self.CF_ALPHA_4_BIT,
                "alpha_8": self.CF_ALPHA_8_BIT,
                "indexed_1": self.CF_INDEXED_1_BIT,
                "indexed_2": self.CF_INDEXED_2_BIT,
                "indexed_4": self.CF_INDEXED_4_BIT,
                "indexed_8": self.CF_INDEXED_8_BIT,
                "raw": self.CF_RAW,
                "raw_alpha": self.CF_RAW_ALPHA,
                "raw_chroma": self.CF_RAW_CHROMA,
            }
            
            cf = format_map.get(self.cf_string, self.CF_TRUE_COLOR_565)
            alpha = "alpha" in self.cf_string
            
            self.convert(cf, alpha)
            c_content = self.format_to_c_array()
            output_cf = cf
        
        # Write file
        with open(filename, 'w') as f:
            f.write(self.get_c_header())
            f.write(c_content)
            f.write(self.get_c_footer(output_cf))
        
        print(f"    C array saved to: {filename}")
    
    def save_binary(self, filename=None, bit_format="565"):
        """Save as binary file"""
        if filename is None:
            filename = f"{self.out_name}.bin"
        
        # Format mapping for binary output
        format_map = {
            "332": self.CF_TRUE_COLOR_332,
            "565": self.CF_TRUE_COLOR_565,
            "565_swap": self.CF_TRUE_COLOR_565_SWAP,
            "888": self.CF_TRUE_COLOR_888,
        }
        
        alpha = "alpha" in self.cf_string
        cf = format_map.get(bit_format, self.CF_TRUE_COLOR_565)
        
        self.convert(cf, alpha)
        
        # LVGL color format mapping for header
        lv_cf_map = {
            self.CF_TRUE_COLOR: 4,
            self.CF_TRUE_COLOR_ALPHA: 5,
            self.CF_TRUE_COLOR_CHROMA: 6,
            self.CF_INDEXED_1_BIT: 7,
            self.CF_INDEXED_2_BIT: 8,
            self.CF_INDEXED_4_BIT: 9,
            self.CF_INDEXED_8_BIT: 10,
            self.CF_ALPHA_1_BIT: 11,
            self.CF_ALPHA_2_BIT: 12,
            self.CF_ALPHA_4_BIT: 13,
            self.CF_ALPHA_8_BIT: 14,
        }
        
        # Determine output format
        if "alpha" in self.cf_string:
            output_cf = self.CF_TRUE_COLOR_ALPHA
        elif "chroma" in self.cf_string:
            output_cf = self.CF_TRUE_COLOR_CHROMA
        else:
            output_cf = self.CF_TRUE_COLOR
        
        lv_cf = lv_cf_map.get(output_cf, 4)
        
        # Create header: format + width + height
        header = lv_cf + (self.w << 10) + (self.h << 21)
        
        with open(filename, 'wb') as f:
            f.write(struct.pack('<I', header))  # Little endian uint32
            f.write(bytes(self.d_out))
        
        print(f"    Binary file saved to: {filename}")


def main():
    """Main function for command line usage"""
    parser = argparse.ArgumentParser(description="LVGL Image Converter")
    parser.add_argument("image", help="Input image file")
    parser.add_argument("-n", "--name", default="image", help="Output name (without extension)")
    parser.add_argument("-f", "--format", choices=["c_array", "bin_332", "bin_565", "bin_565_swap", "bin_888"], 
                       default="c_array", help="Output format")
    parser.add_argument("-cf", "--color-format", 
                       choices=["true_color", "true_color_alpha", "true_color_chroma",
                               "alpha_1", "alpha_2", "alpha_4", "alpha_8",
                               "indexed_1", "indexed_2", "indexed_4", "indexed_8",
                               "raw", "raw_alpha", "raw_chroma"],
                       default="true_color", help="Color format")
    parser.add_argument("-d", "--dithering", action="store_true", help="Enable dithering")
    
    args = parser.parse_args()
    
    try:
        converter = LVGLConverter(args.image, args.name, args.dithering, args.color_format)
        
        if args.format == "c_array":
            converter.save_c_array()
        else:
            bit_format = args.format.replace("bin_", "")
            converter.save_binary(bit_format=bit_format)
            
    except Exception as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)


if __name__ == "__main__":
    main()